//*************************************************************************************************
//
//	Description:
//		leafball_billboard.fx - Shader for billboarded rendering of leafballs & fronds.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship			16/06/2008  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

// Yet another preprocessor hack to special-case the lighting in this shader
#define _USE_HALF_LAMBERTIAN_LIGHTING_

// Tell the lighting backend we want transmissive calcs
// (N.B. This feels messy. Find a better way to do it.)
#define USE_TRANSMISSIVE_LIGHTING
#define _SSAO_READY_
#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_SPECULAR
#define NORMAL_MAPPING
#define CONTROL_MAPPING
#define UV_ANIMATION
#endif

//#undef UV_ANIMATION


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;

float4x4 viewI : ViewI
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(_PS3_)
// Max doesn't support viewproj as an app-supplied parameter
SHARE_PARAM float4x4 worldview : WorldView
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 view : View
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

SHARE_PARAM float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 projMatrix : Projection
<
	bool appEdit = false;
	bool export = false;
>;

#else

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 view : View
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

SHARE_PARAM float4x4 projMatrix : Projection
<
	bool appEdit = false;
	bool export = false;
>;
#endif



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_
// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Second UV holds the billboarding pivot offsets
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
	bool use3dMapping = true;
> = 0;

// Vertex colour channel
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;
#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )							// Specular colour in RGB, shininess in alpha
#endif

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif


// Control map
SPECIALISATION_PARAM( useControlMap, "Use control map?", "CONTROL_MAPPING" )	// TRUE if a control map is to be used for extra effects

#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
DEPENDENT_TEXTURE_PARAM( controlTexture, "Control Tex {UV1}", useControlMap )
#endif

SPECIALISATION_PARAM( useUVAnimation, "UV animation?", "UV_ANIMATION" )

#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
// UV animation constants
float currentTime : TIME
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
> = 0.0f;

// multiplied by vertpos dot to influence the scale of random variations
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vertPhaseMultiplier, "Vert phase mult", useUVAnimation, 0.0f, 16.0f, 0.001f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxAnimatedUVDisplacement, "UV Wind Strength", useUVAnimation, 0.0f, 1.0f, 0.0f )
DEPENDENT_FLOAT4_PARAM_DEFAULTED( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW, "UV Freq U0", "UV Freq V0", "UV Freq U1", "UV Freq V1", useUVAnimation, 0.1f, 0.1f, 1.0f, 1.0f  )
DEPENDENT_FLOAT4_PARAM_DEFAULTED( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW, "UV Ampl U0", "UV Ampl V0", "UV Ampl U1", "UV Ampl V1", useUVAnimation, 1.0f, 1.0f, 0.25f, 0.25f )
#endif	// defined( _3DSMAX_ ) || defined( UV_ANIMATION )

float4 transmissiveMult
<
	string UIName = "Trans Mult";
	string UIWidget = "Spinner";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
	bool export = true;
> = { 0.8f, 1.0f, 0.2f, 1.0f };

float ballRadius
<
	string UIName = "Ball Radius";
	string UIWidget = "Spinner";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
	bool export = true;
> = 0.0f;


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
sampler2D controlMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="controlTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < controlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 billboardPivot: TEXCOORD1;
	float3 colour   : TEXCOORD2;													// Vertex colour
	float3 alpha		: TEXCOORD3;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
	float3 billboardPivot: TEXCOORD1;
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// Texture coords
	float3 billboardPivot: TEXCOORD1;
};

// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed				: TEXCOORD3;												// Noise seed
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 5 )
#endif
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed					: TEXCOORD2;
#endif
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
	float4 coords				: TEXCOORD1;
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed					: TEXCOORD2;
#endif
};


//-----------------------------------------------------------------------
//
// Functions
//

// Cut down version of the usual function, as we're always free-locking leafballs
float3 DoBillboardingNoLock( float3 _inputPosition, float3 _billboardPivot, float4x4 _worldview )
{
	#ifdef _3DSMAX_
		_billboardPivot.y += 1;
		float3 pivotPoint = _inputPosition - _billboardPivot;
	#else
		float3 pivotPoint = _inputPosition - _billboardPivot.xzy;
	#endif

	float3 result = mul( float4( pivotPoint, 1.0f ), _worldview ).xyz;
	
	_billboardPivot.z = -_billboardPivot.z;
	result += _billboardPivot.xyz;

	return result;
}


float4 SmoothCurve( float4 _in )
{
// N.B. The following line doesn't compile sometimes (e.g. in max)
//	return _in * _in * ( 3.0f - ( _in * 2.0f ) );
// But this version does.
	return _in * _in * ( 3.0f + ( ( _in * 2.0f ) * -1.0f ) );
}

float4 TriangleWave( float4 _in )
{
	return abs( frac( _in + 0.5f ) * 2.0f - 1.0f );
}

float4 SmoothTriangleWave( float4 _in )
{
	return SmoothCurve( TriangleWave( _in ) );
}


float2 CalculateAnimatedUVOffset( float2 _maxOffsets, float4 _seeds, float4 _frequencies, float4 _amplitudes )
{
	float4 waves = ( frac( _seeds * _frequencies ) ) * 2.0f - 1.0f;
	waves = SmoothTriangleWave( waves ) * _amplitudes;

	float2 wavesSum = waves.xy + waves.zw;

	return wavesSum * _maxOffsets;
}



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT LeafballVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldview = mul( world, view );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif

#ifdef _3DSMAX_
		// By default, Max flips the V coord of texture channels. This is because it
		// uses a BL coord system, instead of D3D's TL system. Flipping works fine for
		// wrapped textures, but for clamped textures it means the default 0...1 texture
		// space rectangle is off the object. Using 1-v instead of -v is correct in
		// both cases so we have this piece of code here, and the colorchannel declaration
		// above to do things correctly.
		_output.texCoord = float2( _input.texCoord.x, 1.0f - _input.texCoord.y );
#else
		_output.texCoord = _input.texCoord;
#endif

	// billboarding	
	float3 viewPos = DoBillboardingNoLock( _input.position, _input.billboardPivot.xyz, worldview );

	// Calculate vert's world position
	float3 worldPos = mul( float4( viewPos, 1.0f ), viewI ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4( worldEyeVec, 0.0f );
	float3 normalisedEyeVec = normalize( worldEyeVec );

	// We just point the normal towards the eye
	float3 normal = normalisedEyeVec;
	_output.normal = normal;

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// And the tangent space vectors always point in the same direction (the leafball is always facing the viewer)
		_output.tangent = mul( float4( 0.0f, 1.0f, 0.0f, 0.0f ), viewI ).xyz;
		_output.binormal  = mul( float4( 1.0f, 0.0f, 0.0f, 0.0f ), viewI ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		// Animation phases based on object and vertex positions
		float objectPhase = dot( world[ 3 ].xyz, 1.0f );
		float vertexPhase = dot( _input.position.xyz, objectPhase ) * vertPhaseMultiplier;
		_output.seed = currentTime + float2( objectPhase, vertexPhase );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 0.0f;
	float maxSpecPowerValue = 0.0f;
#else
	DEPENDENT_CODE_START( useSpecular )
	float globalSpecularFactorValue = globalSpecularFactor;
	DEPENDENT_CODE_END( useSpecular )
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}




SHADOWGEN_VSOUTPUT LeafballShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldview = mul( world, view );
#endif

	// billboarding	
	float3 viewPos = DoBillboardingNoLock( _input.position, _input.billboardPivot.xyz, worldview );

	// Modify the depth of the position towards the viewer by the leafball radius (this causes the
	// billboard to self-shadow from infront of itself)
	viewPos.z -= ballRadius * 1.1f;

	// Calculate vert's world position
//	float3 worldPos = mul( float4( viewPos, 1.0f ), viewI ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		// Animation phases based on object and vertex positions
		float objectPhase = dot( world[ 3 ].xyz, 1.0f );
		float vertexPhase = dot( _input.position.xyz, objectPhase ) * vertPhaseMultiplier;
		_output.seed = currentTime + float2( objectPhase, vertexPhase );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

#if defined(NEEDS_SHADOW_COORDS)
	OUTPUT_SHADOW_COORDS;
#endif	
	_output.texCoord = _input.texCoord;

	return _output;
}



ZPRIMEDOF_VSOUTPUT LeafballZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldview = mul( world, view );
#endif

	// billboarding	
	float3 viewPos = DoBillboardingNoLock( _input.position, _input.billboardPivot.xyz, worldview );

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		// Animation phases based on object and vertex positions
		float objectPhase = dot( world[ 3 ].xyz, 1.0f );
		float vertexPhase = dot( _input.position.xyz, objectPhase ) * vertPhaseMultiplier;
		_output.seed = currentTime + float2( objectPhase, vertexPhase );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	// Calculate clip-space position of the vertex
	_output.coords = _output.position;
	_output.texCoord = _input.texCoord;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float2 seed				: TEXCOORD3;												// Noise seed
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)
#if defined( UV_ANIMATION )
	float2 seed				: TEXCOORD3_centroid;								// Noise seed
#endif

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD6_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


struct SHADOWGEN_PSINPUT
{
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif	
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed					: TEXCOORD2;
#endif
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT LeafballFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	float4 diffuseTexColour;
	float4 specularTexColour;
	float4 transmissiveTexColour;

	//
	// Read control map (or substitute defaults if no control map used)
	//

	float4 controlTexColour;
	DEPENDENT_CODE_START( useControlMap )
#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
		controlTexColour = tex2D( controlMap, _input.texCoord );
#endif
	DEPENDENT_CODE_ELSE( useControlMap )
#if defined( _3DSMAX_ ) || !defined( CONTROL_MAPPING )
		controlTexColour = float4( 0.5f, 0.5f, 1.0f, 1.0f );		// 50% transmissive, average depth, max displacement
#endif
	DEPENDENT_CODE_END( useControlMap )

	//
	// UV animation (if required)
	// 
	float2 UVoffset = float2( 0.0f, 0.0f );
	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		float2 maxDisp = float2( controlTexColour.b * maxAnimatedUVDisplacement, controlTexColour.b * maxAnimatedUVDisplacement );
		float4 masterFreqs = float4( 1.975f, 0.793f, 0.375f, 0.193f );

		DECLARE_MERGED_FLOAT4_PARAM( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW )
		DECLARE_MERGED_FLOAT4_PARAM( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW )

		UVoffset = CalculateAnimatedUVOffset( maxDisp, _input.seed.xxyy, masterFreqs * UVFrequencies, UVAmplitudes );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	// Read offset diffuse colour
	diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );

	// Calculate shadow receiving pseudo-depth offset
	float worldPosDepthOffset = ( ( controlTexColour.g * 2.0f ) - 1.0f ) * ballRadius;

	//
	// Calculate transmissive tex colour
	//
	float transmissiveness = controlTexColour.r;
	transmissiveTexColour = transmissiveMult * diffuseTexColour.g;

	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, _input.texCoord + UVoffset );
#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to zeros (will be optimised out)
		specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

  // Normalise interpolated vector
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord + UVoffset );

		float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Offset the world position towards the eye based on the value from the control
	// map's depth channel & the constant ball size param. This improves the quality
	// of shadow receiving by breaking up the leafball's flat receiver shape.
	worldPos += eye * worldPosDepthOffset;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	// Copy the alpha over & rescale it to make maximum use of available accuracy
	accumulatedColour.w = saturate( diffuseTexColour.w / ( 128.0f / 255.0f ) );
	
	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


PSOUTPUT LeafballShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	//
	// Get flexibility from control map (or substitute defaults if no control map used)
	//

	float flexibility;
	DEPENDENT_CODE_START( useControlMap )
#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
		flexibility = tex2D( controlMap, _input.texCoord ).b;
#endif
	DEPENDENT_CODE_ELSE( useControlMap )
#if defined( _3DSMAX_ ) || !defined( CONTROL_MAPPING )
		flexibility = 1.0f;
#endif
	DEPENDENT_CODE_END( useControlMap )

	//
	// UV animation (if required)
	// 
	float2 UVoffset = float2( 0.0f, 0.0f );
	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		float2 maxDisp = float2( flexibility * maxAnimatedUVDisplacement, flexibility * maxAnimatedUVDisplacement );
		float4 masterFreqs = float4( 1.975f, 0.793f, 0.375f, 0.193f );

		DECLARE_MERGED_FLOAT4_PARAM( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW )
		DECLARE_MERGED_FLOAT4_PARAM( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW )

		UVoffset = CalculateAnimatedUVOffset( maxDisp, _input.seed.xxyy, masterFreqs * UVFrequencies, UVAmplitudes );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

#if !defined(NEEDS_SHADOW_COORDS)
	PSOUTPUT output;
	
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );
	output.Colour = 0;
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );
	
	return output;
#else
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.5f );

	return output;
#endif	
}

PSOUTPUT LeafballZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	//
	// Get flexibility from control map (or substitute defaults if no control map used)
	//

	float flexibility;
	DEPENDENT_CODE_START( useControlMap )
#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
		flexibility = tex2D( controlMap, _input.texCoord ).b;
#endif
	DEPENDENT_CODE_ELSE( useControlMap )
#if defined( _3DSMAX_ ) || !defined( CONTROL_MAPPING )
		flexibility = 1.0f;
#endif
	DEPENDENT_CODE_END( useControlMap )

	//
	// UV animation (if required)
	// 
	float2 UVoffset = float2( 0.0f, 0.0f );
	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		float2 maxDisp = float2( flexibility * maxAnimatedUVDisplacement, flexibility * maxAnimatedUVDisplacement );
		float4 masterFreqs = float4( 1.975f, 0.793f, 0.375f, 0.193f );

		DECLARE_MERGED_FLOAT4_PARAM( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW )
		DECLARE_MERGED_FLOAT4_PARAM( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW )

		UVoffset = CalculateAnimatedUVOffset( maxDisp, _input.seed.xxyy, masterFreqs * UVFrequencies, UVAmplitudes );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );

	return output;
}



//
// Low Detail Shaders
//

struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;											// UV coords for texture channel 0
};

VSOUTPUT_LD LeafballLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldview = mul( world, view );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// billboarding	
	float3 viewPos = DoBillboardingNoLock( _input.position, _input.billboardPivot.xyz, worldview );

	// Calculate vert's world position
	float3 worldPos = mul( float4( viewPos, 1.0f ), viewI ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	float3 normalisedEyeVec = normalize( worldEyeVec );

	// We just point the normal towards the eye
	float3 normal = normalisedEyeVec;

	// Do vertex lighting
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

PSOUTPUT LeafballLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Apply vertex lighting to base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Copy the alpha over & rescale it to make maximum use of available accuracy
	accumulatedColour.w = saturate( diffuseTexColour.w / ( 128.0f / 255.0f ) );

	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );
	
	return _output;
}




//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Leafball
<
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	bool isBillboard = true;

	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Leafball";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour = "ERMB_RENDER";
	string zprimeDOFTechnique = "_Leafball_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_RENDER";
	string shadowGenTechnique = "_Leafball_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "_Leafball_LowDetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 128;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = true;
		AlphaTestEnable = true;
		AlphaRef = 128;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx LeafballVertexShader();
		PixelShader = compile sce_fp_rsx LeafballFragmentShader();
#else
		VertexShader = compile vs_3_0 LeafballVertexShader();
		PixelShader = compile ps_3_0 LeafballFragmentShader();
#endif
	}
}


technique _Leafball_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 128 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx LeafballShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx LeafballShadowGenFragmentShader();
#elif defined(_XBOX)
			VertexShader = compile vs_3_0 LeafballShadowGenVertexShader();
			PixelShader = compile ps_3_0 LeafballShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 LeafballShadowGenVertexShader();
			PixelShader = compile ps_3_0 LeafballShadowGenFragmentShader();
#endif
	}
}

technique _Leafball_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 128);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx LeafballZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx LeafballZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 128;
	    AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 LeafballZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 LeafballZPrimeDOFFragmentShader();
#endif
	}
}


technique _Leafball_LowDetail
<
  bool preservesGlobalState = true;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx LeafballLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx LeafballLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 LeafballLowDetailVertexShader();
		PixelShader = compile ps_3_0 LeafballLowDetailFragmentShader();
#endif
	}
}
